<?php
/* Copyright (C) 2004-2014	Laurent Destailleur	<eldy@users.sourceforge.net>
 * Copyright (C) 2005-2012	Regis Houssin		<regis.houssin@capnetworks.com>
 * Copyright (C) 2008		Raphael Bertrand	<raphael.bertrand@resultic.fr>
 * Copyright (C) 2010-2014	Juanjo Menent		<jmenent@2byte.es>
 * Copyright (C) 2012      	Christophe Battarel <christophe.battarel@altairis.fr>
 * Copyright (C) 2012       Cédric Salvador     <csalvador@gpcsolutions.fr>
 * Copyright (C) 2012-2014  Raphaël Doursenaud  <rdoursenaud@gpcsolutions.fr>
 * Copyright (C) 2015       Marcos García       <marcosgdf@gmail.com>
 * Copyright (C) 2017-2019  Open-DSI            <support@open-dsi.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 * or see http://www.gnu.org/
 */

/**
 *	\file       htdocs/core/modules/societe/doc/doc_account_statut_csv.modules.php
 *	\ingroup    societe
 *	\brief      File of class to generate customers account statut from account_statut_csv model
 */

require_once DOL_DOCUMENT_ROOT.'/core/lib/company.lib.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/functions2.lib.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/date.lib.php';
require_once DOL_DOCUMENT_ROOT.'/core/modules/export/export_csv.modules.php';
require_once DOL_DOCUMENT_ROOT.'/compta/facture/class/facture.class.php';


/**
 *	Class to manage CSV account statut template account_statut_csv
 */
class doc_account_statut_csv
{
    var $db;
    var $name;
    var $description;
    var $type;

    var $phpmin = array(4,3,0); // Minimum version of PHP required by module
    var $version = 'dolibarr';

	var $emetteur;	// Objet societe qui emet


	/**
	 *	Constructor
	 *
	 *  @param		DoliDB		$db      Database handler
	 */
	function __construct($db)
	{
		global $conf,$langs,$mysoc;

		$langs->load("main");
		$langs->load("extraitcompteclient@extraitcompteclient");

		$this->db = $db;
		$this->name = "account_statut_csv";
		$this->description = $langs->trans('ExtraitCompteClientCSVAccountStatutDescription');

		$this->type = 'csv';

		// Get source company
		$this->emetteur=$mysoc;
		if (empty($this->emetteur->country_code)) $this->emetteur->country_code=substr($langs->defaultlang,-2);    // By default, if was not defined
    }


	/**
     *  Function to build pdf onto disk
     *
     *  @param		Object		$object				Object to generate
     *  @param		Translate	$outputlangs		Lang output object
     *  @param		string		$srctemplatepath	Full path of source filename for generator using a template file
     *  @param		int			$hidedetails		Do not show line details
     *  @param		int			$hidedesc			Do not show desc
     *  @param		int			$hideref			Do not show ref
     *  @return     int         	    			1=OK, 0=KO
	 */
	function write_file($object,$outputlangs,$srctemplatepath='',$hidedetails=0,$hidedesc=0,$hideref=0)
	{
		global $user,$langs,$conf,$mysoc,$db,$hookmanager;

		if (! is_object($outputlangs)) $outputlangs=$langs;

		$outputlangs->load("main");
		$outputlangs->load("extraitcompteclient@extraitcompteclient");

        $date_start = $object->context['account_statut']['date_start'];
        $date_end = $object->context['account_statut']['date_end'];

        $export_type = $object->context['account_statut']['export_type'];

        if ($conf->societe->multidir_output[$object->entity])
		{
			// Definition of $dir and $file
			if ($object->specimen)
			{
				$dir = $conf->societe->dir_output;
				$file = $dir . "/SPECIMEN.csv";
			}
			else
			{
				$objectid = dol_sanitizeFileName($object->id);
                $dir = $conf->societe->multidir_output[$object->entity] . "/" . $objectid;
                $datefile = dol_print_date(dol_now(), '%Y-%m-%d');
                if ($export_type == 'Customer')
                {
                    $thirdparty_code = $object->code_client;
                }
                else if ($export_type == 'Supplier')
                {
                    $thirdparty_code = $object->code_fournisseur;
                }
                $export_type_lang = $langs->transnoentitiesnoconv($export_type);
                $file_name = $langs->transnoentitiesnoconv('ExtraitCompteClientPDFAccountStatutFileName', $export_type_lang, $thirdparty_code, $datefile);
                $file_name = trim($file_name);
				$file = $dir . "/" . dol_sanitizeFileName($file_name) . ".csv";
			}
			if (! file_exists($dir))
			{
				if (dol_mkdir($dir) < 0)
				{
					$this->error=$langs->transnoentities("ErrorCanNotCreateDir",$dir);
					return 0;
				}
			}

			if (file_exists($dir))
			{
				// Add pdfgeneration hook
				if (! is_object($hookmanager))
				{
					include_once DOL_DOCUMENT_ROOT.'/core/class/hookmanager.class.php';
					$hookmanager=new HookManager($this->db);
				}
				$hookmanager->initHooks(array('csvgeneration'));
				$parameters=array('file'=>$file,'object'=>$object,'outputlangs'=>$outputlangs);
				global $action;
				$reshook=$hookmanager->executeHooks('beforeCSVCreation',$parameters,$object,$action);    // Note that $action and $object may have been modified by some hooks

                // Columns order
                $array_selected_sorted = array(
                    'date' => 1,
                    'label' => 1,
                    'total_amount' => 1,
                    'amount_rule' => 1,
                    'balance' => 1,
                );

                // Columns labels
                $array_export_fields_label = array(
                    'date' => 'ExtraitCompteClientPDFAccountStatutDate',
                    'label' => 'ExtraitCompteClientPDFAccountStatutLabel',
                    'total_amount' => 'ExtraitCompteClientPDFAccountStatutTotalAmountTTC',
                    'amount_rule' => 'ExtraitCompteClientPDFAccountStatutAmountRuleTTC',
                    'balance' => 'ExtraitCompteClientPDFAccountStatutBalanceTTC',
                );

                $array_types = array();

                if ($export_type == 'Customer')
                {
                    // Get lines of the customer account statut
                    $sql = 'SELECT f.datef AS date,';
                    if ((float) DOL_VERSION < 10) {
                        $sql .= ' f.facnumber AS label,';
                    } else {
                        $sql .= ' f.ref AS label,';
                    }
                    $sql .= ' f.total_ttc AS total_amount,';
                    $sql .= ' f.type AS invoicetype,';
                    $sql .= ' sum(pf.amount) AS amount_payed,';
                    $sql .= ' sum(rc.amount_ttc) AS amount_creditnote,';
                    $sql .= ' sum(rc2.amount_ttc) AS amount_creditused';
                    $sql .= ' FROM '.MAIN_DB_PREFIX.'facture AS f';
                    $sql .= ' LEFT JOIN '.MAIN_DB_PREFIX.'paiement_facture AS pf';
                    $sql .= ' ON pf.fk_facture = f.rowid';
                    $sql .= ' LEFT JOIN ' . MAIN_DB_PREFIX . 'societe_remise_except AS rc';
                    $sql .= ' ON rc.fk_facture_source = f.rowid';
                    $sql .= ' LEFT JOIN ' . MAIN_DB_PREFIX . 'societe_remise_except AS rc2';
                    $sql .= ' ON rc2.fk_facture = f.rowid';
                    $sql .= ' WHERE f.datef >= \''.dol_print_date($date_start, 'dayrfc').'\'';
                    $sql .= ' AND f.datef <= \''.dol_print_date($date_end, 'dayrfc').'\'';
                    $sql .= ' AND f.fk_soc = '.$object->id;
                    $sql .= ' AND f.fk_statut > 0'; // No draft invoice
                    if (!empty($conf->global->FACTURE_DEPOSITS_ARE_JUST_PAYMENTS)) {
                        $sql .= " AND f.type IN (".Facture::TYPE_STANDARD.",".Facture::TYPE_REPLACEMENT.",".Facture::TYPE_CREDIT_NOTE.",".Facture::TYPE_SITUATION.")";
                    } else {
                        $sql .= " AND f.type IN (".Facture::TYPE_STANDARD.",".Facture::TYPE_REPLACEMENT.",".Facture::TYPE_CREDIT_NOTE.",".Facture::TYPE_DEPOSIT.",".Facture::TYPE_SITUATION.")";
                    }
                    $sql .= ' GROUP BY f.rowid';
                }
                else if ($export_type == 'Supplier')
                {
                    // Get lines of the supplier account statut
                    $sql = 'SELECT f.datef AS date,';
                    $sql .= ' f.ref AS label,';
                    $sql .= ' f.total_ttc AS total_amount,';
                    $sql .= ' f.type AS invoicetype,';
                    $sql .= ' sum(pf.amount) AS amount_payed,';
                    $sql .= ' sum(rc.amount_ttc) AS amount_creditnote,';
                    $sql .= ' sum(rc2.amount_ttc) AS amount_creditused';
                    $sql .= ' FROM ' . MAIN_DB_PREFIX . 'facture_fourn AS f';
                    $sql .= ' LEFT JOIN ' . MAIN_DB_PREFIX . 'paiementfourn_facturefourn AS pf';
                    $sql .= ' ON pf.fk_facturefourn = f.rowid';
                    $sql .= ' LEFT JOIN ' . MAIN_DB_PREFIX . 'societe_remise_except AS rc';
                    $sql .= ' ON rc.fk_invoice_supplier_source = f.rowid';
                    $sql .= ' LEFT JOIN ' . MAIN_DB_PREFIX . 'societe_remise_except AS rc2';
                    $sql .= ' ON rc2.fk_invoice_supplier = f.rowid';
                    $sql .= ' WHERE f.datef >= \'' . dol_print_date($date_start, 'dayrfc') . '\'';
                    $sql .= ' AND f.datef <= \'' . dol_print_date($date_end, 'dayrfc') . '\'';
                    $sql .= ' AND f.fk_soc = ' . $object->id;
                    $sql .= ' AND f.fk_statut > 0'; // No draft invoice
                    $sql .= ' GROUP BY f.rowid';
                }

                $resql = $db->query($sql);
                if (!$resql) {
                    $this->error=$db->error();
                    return 0;
                }

                // Set nblignes with the new facture lines content after hook
				$nblignes = $db->num_rows($resql);

				// Create csv instance
				$csv = new ExportCsv($this->db);

				// Open file
				$csv->open_file($file, $outputlangs);

				// Write header
				$csv->write_header($outputlangs);

                $csv->write_title($array_export_fields_label, $array_selected_sorted, $outputlangs, null);

                for ($i = 0; $i < $nblignes; $i++) {
                    $line = $this->db->fetch_object($resql);
                    $line->date = dol_print_date(dol_stringtotime($line->date), 'day', false, $outputlangs);

                    // 1st column - Total Amount
                    $total_amount = $line->total_amount;
                    $line->total_amount = price($total_amount, 0, $outputlangs);

                    // 2nd column - Total payed
                    if ($line->invoicetype <> 3) {
                        $amount_payed = $line->amount_payed - $line->amount_creditnote + $line->amount_creditused /*+ $line->amount_depositused */;
                    } else {
                        $amount_payed = $line->amount_payed + $line->amount_creditused;
                    }
                    $line->amount_rule = price($amount_payed, 0, $outputlangs);

                    // 3rd column - Total Remain to pay
                    $balance = $total_amount - $amount_payed;
                    $line->balance = price($balance, 0, $outputlangs);

                    $csv->write_record($array_selected_sorted, $line, $outputlangs, $array_types);
                }

                // Write footer
                $csv->write_footer($outputlangs);

                // Close file
                $csv->close_file();

				// Add pdfgeneration hook
				$hookmanager->initHooks(array('pdfgeneration'));
				$parameters=array('file'=>$file,'object'=>$object,'outputlangs'=>$outputlangs);
				global $action;
				$reshook=$hookmanager->executeHooks('afterPDFCreation',$parameters,$this,$action);    // Note that $action and $object may have been modified by some hooks

				if (! empty($conf->global->MAIN_UMASK))
				@chmod($file, octdec($conf->global->MAIN_UMASK));

				return 1;   // No error
			}
			else
			{
				$this->error=$langs->transnoentities("ErrorCanNotCreateDir",$dir);
				return 0;
			}
		}
		else
		{
			$this->error=$langs->transnoentities("ErrorConstantNotDefined","SOC_OUTPUTDIR");
			return 0;
		}
		$this->error=$langs->transnoentities("ErrorUnknown");
		return 0;   // Erreur par defaut
	}


}

