<?php
/* Copyright (C) 2007-2015 Laurent Destailleur  <eldy@users.sourceforge.net>
 * Copyright (C) 2017      Open-DSI             <support@open-dsi.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 *   	\file       htdocs/extraitcompteclient/class/action_extraitcompteclient.class.php
 *		\ingroup    extraitcompteclient
 *		\brief
 */

class ActionsExtraitCompteClient
{
    /**
     * @var DoliDB Database handler.
     */
    public $db;
    /**
     * @var string Error
     */
    public $error = '';
    /**
     * @var array Errors
     */
    public $errors = array();

    /**
     * @var array Hook results. Propagated to $hookmanager->resArray for later reuse
     */
    public $results = array();

    /**
     * @var string String displayed by executeHook() immediately after return
     */
    public $resprints;

    /**
     * Constructor
     *
     * @param        DoliDB $db Database handler
     */
    public function __construct($db)
    {
        $this->db = $db;
    }

    /**
     * Overloading the doActions function : replacing the parent's function with the one below
     *
     * @param   array()         $parameters     Hook metadatas (context, etc...)
     * @param   CommonObject    &$object        The object to process (an invoice if you are in invoice module, a propale in propale's module, etc...)
     * @param   string          &$action        Current action (if set). Generally create or edit or null
     * @param   HookManager     $hookmanager    Hook manager propagated to allow calling another hook
     * @return  int                             < 0 on error, 0 on success, 1 to replace standard code
     */
    function doActions($parameters, &$object, &$action, $hookmanager)
    {
        global $user, $langs;
        $contexts = explode(':', $parameters['context']);

        if (in_array('thirdpartycard', $contexts)) {
            $confirm = GETPOST('confirm');
            $langs->load('extraitcompteclient@extraitcompteclient');

            if ($action == 'confirm_extraitcompteclient_generate_account_statut' && $confirm != 'yes') {
                $action = '';
            }
            if ($action == 'confirm_extraitcompteclient_generate_account_statut' && $confirm == 'yes' && $user->rights->extraitcompteclient->societe->generate_account_statut) {
                if (!isset($object->id)) $object->fetch($parameters['id']);
                $model = GETPOST('extraitcompteclient_generate_account_statut_model');
                if (empty($model)) $model = 'account_statut';
                $export_type = GETPOST('extraitcompteclient_generate_account_statut_type');
                if (empty($export_type)) $export_type = 'Customer';
                $date_start = dol_mktime(0, 0, 0, GETPOST('extraitcompteclient_generate_account_statut_date_start_month'), GETPOST('extraitcompteclient_generate_account_statut_date_start_day'), GETPOST('extraitcompteclient_generate_account_statut_date_start_year'));
                $date_end = dol_mktime(0, 0, 0, GETPOST('extraitcompteclient_generate_account_statut_date_end_month'), GETPOST('extraitcompteclient_generate_account_statut_date_end_day'), GETPOST('extraitcompteclient_generate_account_statut_date_end_year'));
                if (!isset($object->context)) $object->context = array();
                $object->context['account_statut'] = array('date_start' => $date_start, 'date_end' => $date_end, 'export_type' => $export_type);

                // Generate account statut PDF / CSV
                $result = $object->generateDocument($model, $langs, 0, 0, 0);
                if ($result < 0) {
                    dol_print_error($this->db, $result);
                    $this->errors[] = 'Error generate account statut PDF / CSV';
                    return -1;
                }
            }
        }

        return 0;
    }

    /**
     * Overloading the addMoreActionsButtons function : replacing the parent's function with the one below
     *
     * @param   array()         $parameters     Hook metadatas (context, etc...)
     * @param   CommonObject    &$object        The object to process (an invoice if you are in invoice module, a propale in propale's module, etc...)
     * @param   string          &$action        Current action (if set). Generally create or edit or null
     * @param   HookManager     $hookmanager    Hook manager propagated to allow calling another hook
     * @return  int                             < 0 on error, 0 on success, 1 to replace standard code
     */
    function addMoreActionsButtons($parameters, &$object, &$action, $hookmanager)
    {
        global $conf, $db, $langs, $user;
        $error = 0; // Error counter
        $myvalue = 1; // A result value

        if (in_array('thirdpartycard', explode(':', $parameters['context']))) {
            $form = new Form($db);
            $langs->load('extraitcompteclient@extraitcompteclient');

            $month = isset($conf->global->SOCIETE_FISCAL_MONTH_START) ? intval($conf->global->SOCIETE_FISCAL_MONTH_START) : 1;
            $year = intval(date('n')) < $month ? intval(date('Y')) - 1 : intval(date('Y'));
            $date_start = new DateTime(sprintf('%04d-%02d-%02d', $year, $month, 1));
            $date_end = new DateTime(sprintf('%04d-%02d-%02d', $year, $month, 1));
            $date_end->add(new DateInterval('P1Y'));
            $date_end->sub(new DateInterval('P1D'));

            $models = array(
                'account_statut' => 'PDF',
                'account_statut_csv' => 'CSV',
            );

            $transcustomer = trim($langs->trans('ExtraitCompteClientGenerateAccountStatutTypeCustomer'));
            $transsupplier = trim($langs->trans('ExtraitCompteClientGenerateAccountStatutTypeSupplier'));
            if(! empty($object->client) && empty($object->fournisseur)) {
                $export_type = array(
                    'Customer' => $transcustomer,
                );
                $export_default = 'Customer';
            } else if(empty($object->client) && ! empty($object->fournisseur)) {
                $export_type = array(
                    'Supplier' => $transsupplier,
                );
                $export_default = 'Supplier';
            } else {
                $export_type = array(
                    'Customer' => $transcustomer,
                    'Supplier' => $transsupplier,
                );
                $export_default = 'Customer';
            }


            // Define confirmation messages
            $formquestionclone = array(
                array('type' => 'date', 'name' => 'extraitcompteclient_generate_account_statut_date_start_', 'label' => $langs->trans("ExtraitCompteClientGenerateAccountStatutDateStart"), 'value' => $date_start->format('Y-m-d')),
                array('type' => 'date', 'name' => 'extraitcompteclient_generate_account_statut_date_end_', 'label' => $langs->trans("ExtraitCompteClientGenerateAccountStatutDateEnd"), 'value' => $date_end->format('Y-m-d')),
                array('type' => 'select', 'name' => 'extraitcompteclient_generate_account_statut_type', 'label' => $langs->trans("ExtraitCompteClientGenerateAccountStatutType"), 'values' => $export_type, 'default' => $export_default),
                array('type' => 'select', 'name' => 'extraitcompteclient_generate_account_statut_model', 'label' => $langs->trans("ExtraitCompteClientGenerateAccountStatutModel"), 'values' => $models, 'default' => 'account_statut'),
            );

            // Generate account statut confirmation
            if (($action == 'extraitcompteclient_generate_account_statut' && (empty($conf->use_javascript_ajax) || !empty($conf->dol_use_jmobile)))        // Output when action = extraitcompteclient_generate_account_statut if jmobile or no js
                || (!empty($conf->use_javascript_ajax) && empty($conf->dol_use_jmobile))
            )                                                            // Always output when not jmobile nor js
            {
                print $form->formconfirm($_SERVER["PHP_SELF"] . '?socid=' . $object->id, $langs->trans('ExtraitCompteClientGenerateAccountStatutTitle'), $langs->trans('ExtraitCompteClientGenerateAccountStatutConfirmLabel', $object->ref), 'confirm_extraitcompteclient_generate_account_statut', $formquestionclone, 'yes', 'action-extraitcompteclient_generate_account_statut', 280, 500);
            }

            if ($user->rights->extraitcompteclient->societe->generate_account_statut) {
                if (!empty($conf->use_javascript_ajax) && empty($conf->dol_use_jmobile)) {
                    print '<div class="inline-block divButAction"><span id="action-extraitcompteclient_generate_account_statut" class="butAction">' . $langs->trans('ExtraitCompteClientGenerateAccountStatut') . '</span></div>' . "\n";
                } else {
                    print '<div class="inline-block divButAction"><a class="butAction" href="' . $_SERVER["PHP_SELF"] . '?action=extraitcompteclient_generate_account_statut&amp;socid=' . $object->id . '">' . $langs->trans("EXTRAITCompteClientGenerateAccountStatut") . '</a></div>';
                }
            }
        }

        if (!$error) {
            $this->results = array('myreturn' => $myvalue);
            $this->resprints = '';
            return 0; // or return 1 to replace standard code
        } else {
            $this->errors[] = 'Error ExtraitCompteClient module hook addMoreActionsButtons';
            return -1;
        }
    }
}